/*
 This file is part of GNU Taler
 (C) 2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  NotificationType,
  ObservabilityEventType,
  TransactionMajorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation, parseTransactionIdentifier } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState, getTestHarnessPaytoForLabel } from "harness/harness.js";
import { createSimpleTestkudosEnvironmentV3, withdrawViaBankV3 } from "harness/environments.js";
import { TaskRunResultType } from "../../../taler-wallet-core/src/common.js";

/**
 * Run test for hintNetworkAvailability in wallet-core
 */
export async function runWalletNetworkAvailabilityTest(t: GlobalTestState) {

  // Set up test environment
  const { bankClient, walletClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t, undefined, {
      // We need this to listen to the network-required observability event
      walletTestObservability: true,
    });

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const networkRequiredCond = walletClient.waitForNotificationCond((x) => {
    return (x.type === NotificationType.TaskObservabilityEvent
      && x.event.type === ObservabilityEventType.ShepherdTaskResult
      && x.event.resultType === TaskRunResultType.NetworkRequired
    );
  });

  const refreshCreatedCond = walletClient.waitForNotificationCond((x) => {
    return (x.type === NotificationType.TransactionStateTransition &&
      parseTransactionIdentifier(x.transactionId)?.tag === TransactionType.Refresh
    );
  });

  const refreshDoneCond = walletClient.waitForNotificationCond((x) => {
    return (x.type === NotificationType.TransactionStateTransition &&
      parseTransactionIdentifier(x.transactionId)?.tag === TransactionType.Refresh
      && x.newTxState.major === TransactionMajorState.Done
    );
  });

  await walletClient.call(WalletApiOperation.HintNetworkAvailability, {
    isNetworkAvailable: false,
  });

  const depositGroupResult = await walletClient.client.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:10.5" as AmountString,
      depositPaytoUri: getTestHarnessPaytoForLabel("foo"),
    },
  );

  // refresh should not continue due to network-required
  await networkRequiredCond;

  await walletClient.call(WalletApiOperation.HintNetworkAvailability, {
    isNetworkAvailable: true,
  });

  await refreshCreatedCond;

  // refresh should finish due to network being restored
  await refreshDoneCond;
}
